/*
 * Copyright (c) 2008 Lu, Chao-Ming (Tetralet).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "vtefont.h"

extern gchar *current_font_name;
extern gchar *default_font_name;
// For restore to original font and size
extern gint original_screen_width;
extern gint original_screen_height;

extern GtkWidget *window;
extern GtkWidget *notebook;
extern GtkWidget *current_vtebox;

gint force_resize_window=0;
gboolean update_hints=FALSE;

void set_vtebox_font(GtkWidget *widget, gint type)
{
	// type 0: reset current page's font size
	// type 1: increase current page's font size
	// type 2: decrease current page's font size
	// type 3: reset window size & font size for every vtebox
	// type 4: increase window size & font size for every vtebox
	// type 5: decrease window size & font size for every vtebox
	// type 6: change current page's font name & size
	// type 7: change current page's font name & size for every vtebox
	switch (type)
	{
		case 0:
			// reset current page's font size
			get_resize_font(0);
			reset_vtebox_size(0);
			break;
		case 1:
			// increase current page's font size
			get_resize_font(1);
			reset_vtebox_size(0);
			break;
		case 2:
			// decrease current page's font size
			get_resize_font(2);
			reset_vtebox_size(0);
			break;
		case 3:
			// reset window size & font size for every vtebox
			get_resize_font(0);
			reset_vtebox_size(2);
			break;
		case 4:
			// increase window size & font size for every vtebox
			get_resize_font(1);
			reset_vtebox_size(1);
			break;
		case 5:
			// decrease window size & font size for every vtebox
			get_resize_font(2);
			reset_vtebox_size(1);
			break;
		case 6:
			// change current page's font name & size
			get_resize_font(3);
			reset_vtebox_size(3);
			break;
		case 7:
			// change current page's font name & size for every vtebox
			get_resize_font(3);
			if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook))==1)
				reset_vtebox_size(3);
			else
				reset_vtebox_size(4);
			break;
	}
}

void reset_vtebox_size(gint type)
{
	gint total_page = gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook));
	
	// type 0: change current page's font
	// type 1: increase/decrease window size & font size for every vtebox
	// type 2: reset window size & font size for every vtebox
	// type 3: change current page's font name & size
	// type 4: Change the font & size of every tab

	switch (type)
	{
		case 0:
		{
			// change current page's font
			// g_debug("Trying to apply font %s to vtebox\n", current_font_name);
			vte_terminal_set_font_from_string(VTE_TERMINAL(current_vtebox), current_font_name);
			window_resizable(window, current_vtebox, 2, 1);
			if (total_page !=1)
				update_hints = TRUE;
			break;
		}
		case 1:
		{
			// increase/decrease window size & font size for every vtebox
			// g_debug("Trying to apply font %s to every vtebox\n", current_font_name);
			apply_font_to_every_vtebox(vte_terminal_get_column_count(VTE_TERMINAL(current_vtebox)),
						   vte_terminal_get_row_count(VTE_TERMINAL(current_vtebox)));
			break;
		}
		case 2:
		{
			// reset window size & font size for every vtebox
			// g_debug("Trying to apply font %s to every vtebox\n", current_font_name);
			apply_font_to_every_vtebox(original_screen_width, original_screen_height);
			break;
		}
		case 3:
		{
			// change current page's font name & size
			vte_terminal_set_font_from_string(VTE_TERMINAL(current_vtebox),
							  current_font_name);
			if (total_page ==1)
			{
				// Apply the font and resize whole window to fit vte size
				window_resizable(window, current_vtebox, 1, -1);
				// g_debug("resize widow to 1x1! \n");
				gtk_window_resize(GTK_WINDOW(window), 1, 1);
			}
			else
			{
				window_resizable(window, current_vtebox, 2, 1);
				update_hints = TRUE;
			}
			force_resize_window=1;
			break;
		}
		case 4:
		{
			apply_font_to_every_vtebox(-1, -1);
			window_resizable(window, current_vtebox, 1, 1);
			break;
		}
	}
}

// it will update the current_font_name and current_data->font_name
void get_resize_font(gint type)
{
	// we must insure that vtebox!=NULL
	struct Page *current_data = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");

	// type 0: restore font to default_font_name 
	// type 1: increase
	// type 2: decrease
	// type 3: current_data->font_name = current_font_name
	gint fontsize, oldfontsize;
	// g_debug("old font name: %s\n", current_data->font_name);
	PangoFontDescription *font_desc = pango_font_description_from_string(current_data->font_name);
	
	g_free(current_data->font_name);
	switch (type)
	{
		case 0:
		{
			// restore font to default_font_name
			current_data->font_name = g_strdup(default_font_name);
			break;
		}
		case 1:
		case 2:
		{
			// increase/decrease font
			oldfontsize = (pango_font_description_get_size(font_desc)/PANGO_SCALE);
			if (type==1)
			{
				fontsize = (pango_font_description_get_size(font_desc)/PANGO_SCALE)*1.13;
				if (oldfontsize==fontsize)
					fontsize++;
			}
			else
			{
				fontsize = (pango_font_description_get_size(font_desc)/PANGO_SCALE)/1.1;
				if (oldfontsize==fontsize)
					fontsize--;
				if (fontsize<1)
					fontsize=1;
			}
			// g_debug("Trying to change the font size to %d.\n", fontsize);
			pango_font_description_set_size(font_desc, fontsize*PANGO_SCALE);
			current_data->font_name = pango_font_description_to_string(font_desc);
			break;
		}
	}
	if (type==3)
		// current_data->font_name = current_font_name
		current_data->font_name = current_font_name;
	else
		current_font_name = current_data->font_name;
	// g_debug("new font name: %s\n", current_data->font_name);
}

void apply_font_to_every_vtebox(gint colum, gint row)
{
	GtkWidget *vtebox;
	struct Page *current_data;
	gint i;
	gchar *new_font_name = g_strdup(current_font_name);

	// g_debug("Trying to apply font %s to every vtebox!\n", new_font_name);
	for (i=0;i<gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook));i++)
	{
		vtebox=(GtkWidget *)g_object_get_data(G_OBJECT( gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook),
									gtk_notebook_get_nth_page(
										GTK_NOTEBOOK(notebook), i))),
								"VteBox");
		current_data = (struct Page *)g_object_get_data(G_OBJECT(vtebox), "Data");
		// g_debug("The original font for %d page is: %s (%s)\n", i, current_data->font_name, new_font_name);
		vte_terminal_set_font_from_string(VTE_TERMINAL(vtebox), new_font_name);

		vte_terminal_set_size(VTE_TERMINAL(vtebox), colum, row);
		g_free(current_data->font_name);
		current_data->font_name = g_strdup(new_font_name);
		// g_debug("The new font for %d page is: %s (%s)\n", i, current_data->font_name, new_font_name);
	}
	if (colum>0)
	{
		window_resizable(window, current_vtebox, 1, -1);
		// Yes, just set it to 1 x 1,
		// and the window will be resized to correct geometry automatically.
		gtk_window_resize(GTK_WINDOW(window), 1, 1);
	}
	// g_debug("Set hints to FALSE!\n");
	force_resize_window=1;
	update_hints = FALSE;
	
	g_free(new_font_name);
}
