/*
 * Copyright (c) 2008 Lu, Chao-Ming (Tetralet).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "notebook.h"

// For command line options.
extern gchar *command_line;
extern gchar **parameter;
// For using transparent background
extern gboolean transparent_background;
extern gdouble background_saturation;
// For showing the Menu if right click on a vtebox.
extern GtkWidget *menu;
// For setting the size when adding a new page, avoid to set it to 80x24 (libvte default size)
extern gint screen_width;
extern gint screen_height;
// For sloving the function key
extern struct KeyValue pagekeys[KEYS];
// For increase/decrease the size of font
extern gchar *current_font_name;
// For Transparent Background
extern GtkWidget *trans_bg_item;
extern GtkWidget *adjustment;
// For Set the radio_list to System Defalut
extern GtkWidget *default_encoding;
// For monitor the pid's stat
extern gboolean page_shows_current_cmdline;
extern gboolean window_shows_current_page;

extern gboolean update_hints;

extern GtkWidget *window;
extern GtkWidget *notebook;
GtkWidget *current_vtebox=NULL;

void add_page(gint run_once)
{
	// the component of a single page
	struct Page *new_page = g_new0(struct Page, 1);
	const gchar *directory = g_getenv("PWD");

	// create label
	new_page->label = gtk_label_new(NULL);
	gtk_label_set_max_width_chars (GTK_LABEL(new_page->label), 20);
	gtk_label_set_ellipsize(GTK_LABEL(new_page->label), PANGO_ELLIPSIZE_MIDDLE);

	// create a hbox
	new_page->hbox = gtk_hbox_new(FALSE, 0);

	// create vtebox
	new_page->vtebox = vte_terminal_new();

	// Get current vtebox size, font name and directory. for init a new tab.
	if ( gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook)) != -1 && current_vtebox != NULL)
	{
		screen_width = vte_terminal_get_column_count(VTE_TERMINAL(current_vtebox));
		screen_height = vte_terminal_get_row_count(VTE_TERMINAL(current_vtebox));
		// g_debug("Trying to fix vtebox to %d x %d when showing the tab bar\n",
		//	   screen_width, screen_height);

		// Got the current page's directory
		struct Page *prev_page = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");
		directory = g_file_read_link(g_strdup_printf("/proc/%d/cwd", prev_page->pid), NULL);
		// Got the default font name
		new_page->font_name = g_strdup(prev_page->font_name);
		current_font_name = new_page->font_name;
		// g_debug("use the directory %s (/proc/%d/cwd)\n", directory, prev_page->pid);
	}
	else
		new_page->font_name = g_strdup(current_font_name);

	// Init new page. run_once: some settings only need run once.
	// run_once only = TRUE when init LilyTerm in main().
	// init_new_page(window, new_page->vtebox, run_once);
	gtk_box_pack_start(GTK_BOX(new_page->hbox), new_page->vtebox, TRUE, TRUE, 0);
	// the close page event
	g_signal_connect(G_OBJECT(new_page->vtebox), "child_exited",
			 G_CALLBACK(close_page), FALSE);
	// when get focus, update `current_vtebox'
	g_signal_connect(G_OBJECT(new_page->vtebox), "grab-focus",
			 G_CALLBACK(page_grab_focuse), NULL);
// #ifdef HINTS
	// when vtebox is focuse, the size of vtebox is unchangeable.
//	g_signal_connect_after(G_OBJECT(new_page->vtebox), "focus-in-event", G_CALLBACK(page_get_focuse), NULL);
	// g_signal_connect_after(G_OBJECT(new_page->vtebox), "lower-window", G_CALLBACK(page_get_focuse), NULL);
	// when vtebox is focuse, the size of vtebox is changeable.
//	g_signal_connect(G_OBJECT(new_page->vtebox), "focus-out-event", G_CALLBACK(page_lost_focuse), NULL);
//	g_signal_connect_after(G_OBJECT(new_page->vtebox), "size-allocate", G_CALLBACK(vtebox_size_allocate), NULL);
//	g_signal_connect(G_OBJECT(new_page->vtebox), "size-request", G_CALLBACK(vtebox_size_request), NULL);
//	g_signal_connect_after(G_OBJECT(new_page->vtebox), "style-set", G_CALLBACK(vtebox_style_set), NULL);
// #endif
	// if function key pressed
	g_signal_connect(G_OBJECT(new_page->vtebox), "key-press-event",
			 G_CALLBACK(vtebox_key_press), NULL);
	// show the menu
	g_signal_connect(G_OBJECT(new_page->vtebox), "button-press-event",
			 G_CALLBACK(vtebox_button_press), NULL);

	// scrollbar
	new_page->scrollbar = gtk_vscrollbar_new(vte_terminal_get_adjustment(VTE_TERMINAL(new_page->vtebox)));
	gtk_box_pack_start(GTK_BOX(new_page->hbox), new_page->scrollbar, FALSE, FALSE, 0);

	// add the new page to notebook
	new_page->current_page_no = gtk_notebook_append_page(GTK_NOTEBOOK(notebook), new_page->hbox, new_page->label);

#ifdef ENABLE_TAB_REORDER
	gtk_notebook_set_tab_reorderable(GTK_NOTEBOOK(notebook), new_page->hbox,TRUE);
	gtk_notebook_set_tab_detachable(GTK_NOTEBOOK(notebook), new_page->hbox,FALSE);
#endif
	g_object_set(gtk_widget_get_parent(new_page->label), "can-focus", FALSE, NULL);

	// show the page bar if necessary
	if (new_page->current_page_no == 1)
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), TRUE);

	// Execute programs in the vtebox
	//if (parameter != NULL)
	//	g_debug ("Run Command: %s %s\n", command_line, *parameter);
#ifdef GOT_SHELL_COMMAND
	if (command_line==NULL)
		command_line = g_getenv("SHELL");
#endif
	new_page->pid = vte_terminal_fork_command(VTE_TERMINAL(new_page->vtebox),
						  command_line, parameter, NULL,
						  directory, TRUE, TRUE, TRUE);
	// We MUST clear command_line and parameter after run -e option.
	command_line = NULL;
	parameter = NULL;

	// for debug only
	// gchar *pid = g_strdup_printf("%d", new_page->pid);
	// gtk_label_set_text(GTK_LABEL(new_page->label), pid);
	// g_free(pid);

	// set the tab name.
	// we store stat_path here for performance.
	new_page->stat_path = g_strdup_printf("/proc/%d/stat", (gint) new_page->pid);
	// we need to g_free it in update_tab_name(). so we need to set it to NULL.
	new_page->label->name = NULL;
	new_page->custom_page_name = NULL;
	update_tab_name(new_page->stat_path, new_page->label, new_page->pid, &(new_page->tpgid),
			new_page->current_page_no + 1, new_page->custom_page_name);
	// g_debug("Got label name from update_tab_name(): %s\n", new_page->label->name);

	// Monitor cmdline
	if (page_shows_current_cmdline)
		// monitor_cmdline(new_page->monitor, new_page->pid);
		// monitor_cmdline(new_page->channel, new_page->pid);
#ifdef USE_TIMEOUT_SECONDS
		new_page->timeout_id = g_timeout_add_seconds (1, (GSourceFunc)monitor_cmdline, NULL);
#else
		new_page->timeout_id = g_timeout_add (1000, (GSourceFunc)monitor_cmdline, NULL);
#endif
	// finish!
	gtk_widget_show_all(new_page->hbox);
	// init new page after the whole window is shown
	// run_once: some settings only need run once.
	// run_once only = TRUE when init LilyTerm in main().
	init_new_page(window, new_page->vtebox, run_once);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), new_page->current_page_no);
	gtk_window_set_focus(GTK_WINDOW(window), new_page->vtebox);
	
	// Store the new page data
	new_page->encoding = default_encoding;
	new_page->transparent_background = transparent_background;
	new_page->background_saturation = background_saturation;
	g_object_set_data(G_OBJECT(new_page->vtebox), "Data", new_page);
	g_object_set_data(G_OBJECT(new_page->label), "VteBox", new_page->vtebox);
}

gboolean close_page (GtkWidget *widget, gboolean need_safe_close)
{
	gint total_page = gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook));
	// gint current_page_no = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	// g_debug("Total Page (Notebook):%d\n", total_page);

	struct Page *current_data = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");
	// g_debug("Deleting Page ID: %d\n", current_data->pid);

	if (need_safe_close)
	{
		gint tpgid = get_tpgid(current_data->stat_path, current_data->pid);
		// g_debug("pid=%d, and tpgid=%d\n", current_data->pid, tpgid);
		if (current_data->pid != tpgid)
			if (dialog(NULL, 7)==FALSE) return FALSE;
	}

	// 
	if (page_shows_current_cmdline)
	//	monitor_cmdline_cancel(current_data->monitor);
		g_source_remove (current_data->timeout_id);

	// stop running shell
	if (need_safe_close)
	{
		// g_debug("Trying to kill %d!\n", current_data->pid);
		kill(current_data->pid, SIGKILL);
	}

	// if it is the last page, exit application!
	if (total_page == 1)
	{
		// g_debug("It is the last page!\n");
		gtk_main_quit();
	}
	else
	{
		// set the hints to next vtebox!
		// or sometimes the window size will change when close current page
		GtkWidget *vtebox;
		if ( current_data->current_page_no < total_page-1 )
		{
			vtebox = (GtkWidget *)g_object_get_data(
					G_OBJECT( gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook),
						  gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook),
							current_data->current_page_no+1))),
					"VteBox");
		}
		else
		{
			vtebox = (GtkWidget *)g_object_get_data(
					G_OBJECT( gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook),
						  gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook),
							current_data->current_page_no-1))),
					"VteBox");
		}
		window_resizable(window, vtebox, 0, -1);
		
		// remove current page
		// g_debug ("The %d page is going to be removed!\n", current_data->current_page_no);
		// use current_data->current_page_no. DANGEROUS!
		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), current_data->current_page_no);
		
		// set the current page
		// g_debug ("Setting current page to %d!\n", current_data->current_page_no);
		if ( current_data->current_page_no < total_page-1 )
			gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), current_data->current_page_no);
		
		// hide the page bar if necessary
		if (total_page == 2)
		{
			// hide the page bar
			gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
			// Yes, just set it to 1 x 1,
			// and the window will be resized to correct geometry automatically.
			gtk_window_resize(GTK_WINDOW(window), 1, 1);
		}
	}
	// if (page_number && (current_data->current_page_no < total_page-1))
	if (current_data->current_page_no < total_page-1)
		reorder_page_number(NULL, NULL);

	// free the memory used by this page
	// g_debug("freeing current_data!\n");
	g_free(current_data->stat_path);
	g_free(current_data);
	return TRUE;
}

void page_grab_focuse(GtkWidget *vtebox, gpointer user_data)
{
	// g_debug ("Update current_vtebox!\n");
	current_vtebox = vtebox;
	
	// g_debug("Updating hints for this page!\n");
	if (update_hints)
		// if update_hints = TRUE, we should update the font hints for every vtebox.
		window_resizable(window, current_vtebox, 1, -1);
	else
		// else, updte the hints without font
		window_resizable(window, current_vtebox, 0, -1);

	if (window_shows_current_page)
	{
		struct Page *current_data = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");
		if (current_data!=NULL)
			update_window_title(current_data->label->name);
	}
}

gboolean vtebox_button_press(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	if (event->button == 3)
	{
		struct Page *current_data = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(current_data->encoding), TRUE);
		GTK_CHECK_MENU_ITEM(trans_bg_item)->active = current_data->transparent_background;
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, event->button, event->time);
		return TRUE;
	}
	return FALSE;
}

gboolean vtebox_key_press(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
	// don't check if no ctrl/shift/alt key pressed!
	if (event->state & ALL_ACCELS_MASK)
	{
		// don't check if only shift key pressed!
		if (event->state & SHIFT_ONLY_MASK)
		{
			// g_debug("ALL_ACCELS_MASK = %X\n", ALL_ACCELS_MASK);
			// g_debug("Got the function key: %s (status: %X, check: %X)\n",
			//	   gdk_keyval_name(event->keyval), event->state,
			//	   event->state|GDK_LOCK_MASK|GDK_MOD2_MASK);
			gint i, keyval, mods;
			if ((event->keyval>=GDK_a) && (event->keyval<=GDK_z))
				keyval = event->keyval - GDK_a + GDK_A;
			else
				keyval = event->keyval;
			mods = event->state|GDK_LOCK_MASK|GDK_MOD2_MASK;
			for (i=0;i<KEYS;i++)
			{
				// g_debug("Checking... key: %s (status: %X)\n",
				//	   gdk_keyval_name(pagekeys[i].key), event->state|GDK_LOCK_MASK);
				if ((mods==pagekeys[i].mods) && (keyval==pagekeys[i].key))
				{
					// deal the function key
					deal_key_press(i);
					return TRUE;
				}
			}
		}
	}
	return FALSE;
}

// Default function key:
//  [0] <Ctrl><T> = New Page
//  [1] <Ctrl><W> = Close Page
//  [2] <Ctrl><E> = Edit Page Label
//  [3] <Ctrl><V> = Past Page Label
//  [4] <Ctrl><PageUp> = Pre Page
//  [5] <Ctrl><PageDown> = Next Page
//  [6] <Ctrl><Home> = First Page
//  [7] <Ctrl><End> = Last Page
//  [8] <Ctrl><E> = Move Page Forward
//  [9] <Ctrl><C> = Move Page Backward
// [10] <Ctrl><E> = Move Page to First
// [11] <Ctrl><C> = Move Page to Last
//		<Shift><Insert> to pase clipboard.
void deal_key_press(gint type)
{
	// gint current_page_no = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	gint total_page = gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook));
	struct Page *current_data = (struct Page *)g_object_get_data(G_OBJECT(current_vtebox), "Data");

	switch (type)
	{
		case 0:
		{
			// add a new page
			add_page(0);
			break;
		}
		case 1:
		{
			// close page
			// g_debug("Trying to close page!\n");
			close_page (NULL, TRUE);
			break;
		}
		case 2:
		{
			// edit page's label
			dialog(NULL, 1);
			break;
		}
		case 3:
		{
			// past page's label
			// g_debug("Trying to rename the label whith the selected text.\n");
			break;
		}
		case 4:
		{
			// switch to pre page
			if (current_data->current_page_no)
				gtk_notebook_prev_page(GTK_NOTEBOOK(notebook));
			else
				gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), total_page -1);
			break;
		}
		case 5:
		{
			// switch to next page
			if (current_data->current_page_no == (gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook)) - 1))
				gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 0);
			else
				gtk_notebook_next_page(GTK_NOTEBOOK(notebook));
			break;
		}
		case 6:
		{
			// switch to first page
			gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 0);
			break;
		}
		case 7:
		{
			// switch to last page
			gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), total_page-1);
			break;
		}
		case 8:
		{
			// move current page forward
			gtk_notebook_reorder_child(GTK_NOTEBOOK(notebook), current_data->hbox, 
						   current_data->current_page_no -1);
			break;
		}
		case 9:
		{
			// move current page backward
			if (current_data->current_page_no == (gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook))-1))
				 gtk_notebook_reorder_child(GTK_NOTEBOOK(notebook), current_data->hbox, 0);
			else
				gtk_notebook_reorder_child(GTK_NOTEBOOK(notebook),
							   current_data->hbox, current_data->current_page_no+1);
			break;
		}
		case 10:
		{
			// move current page to first
			gtk_notebook_reorder_child(GTK_NOTEBOOK(notebook), current_data->hbox, 0);
			break;
		}
		case 11:
		{
			// move current page to last
			gtk_notebook_reorder_child(GTK_NOTEBOOK(notebook), current_data->hbox, -1);
			break;
		}
		case 12:
		{
			set_vtebox_font(NULL, 1);
			break;
		}
		case 13:
		{
			set_vtebox_font(NULL, 2);
			break;
		}
		case 14:
		{
			set_vtebox_font(NULL, 0);
			break;
		}
		case 15:
		case 16:
		case 17:
		case 18:
		case 19:
		case 20:
		case 21:
		case 22:
		case 23:
		case 24:
		case 25:
		case 26:
		{
			// switch to #%d page
			if (total_page > type-15)
				gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), type-15);
			break;
		}
		case 27:
		{
#ifdef ENABLE_SELECT_ALL
			vte_terminal_select_all(VTE_TERMINAL(current_vtebox));
#endif
			break;
		}
		case 28:
		{
			vte_terminal_copy_clipboard(VTE_TERMINAL(current_vtebox));
			break;
		}
		case 29:
		{
			vte_terminal_paste_clipboard(VTE_TERMINAL(current_vtebox));
			break;
		}
	}
}
